/*
 * This file is part of TileLabels, which is a NoGo Library for OpenTTD
 * Copyright (C) 2012  Leif Linse
 *
 * TileLabels is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * TileLabels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TileLabels; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class TileLabels
{
	tile_table = null;
	prefix = null;

	/*
	 * prefix: A string that signs must start with to be used as a
	 * label sign. All signs that start with this prefix will be
	 * removed from the map when they are read by ReadFromMap, so
	 * pick a prefix that don't clash with informative labels that
	 * you intend your players to see. 
	 *
	 * You must not pass null, but an empty string can be used if you 
	 * don't want to have a prefix. Note 
	 * 
	 */
	constructor(prefix = "$L=")
	{
		this.prefix = prefix;
		this.tile_table = {};
	}

	/*
	 * Takes a label (string) and returns the tile index
	 * of the tile that contained a sign for this label.
	 * Note that you only pass the label string and not the
	 * "L:" prefix.
	 *
	 * If no matching label exist, -1 is returned. Preferable
	 * use GSMap.IsValidTile to check the return value.
	 */
	function GetTile(label);

	/*
	 * Reads tile labels from the map, and then remove the label signs.
	 *
	 * Ideally call this function in your Start() function during
	 * the initialization of your Game Script.
	 */
	function ReadFromMap();

	/*
	 * Returns a squirrel table that can be stored
	 * in the savegame table.
	 */
	function ExportToSave();

	/*
	 * Pass the table that ExportToSave() returned which you have
	 * stored in a save game. Eg. don't pass your entire table
	 * that you get from OpenTTD, only the part of it that 
	 * correspond to what you got from ExportToSave().
	 */
	function ImportFromSave(save_data);
}

function TileLabels::GetTile(label)
{
	if (this.tile_table.rawin(label))
		return this.tile_table.rawget(label);

	return -1;
}

function TileLabels::ReadFromMap()
{
	local sign_list = GSSignList();
	foreach (sign, _ in sign_list)
	{

		local sign_content = GSSign.GetName(sign);
		local pre_part = sign_content.slice(0, this.prefix.len());

		// Does the sign start with the required prefix?
		if(pre_part == this.prefix)
		{
			// Add tile label to tile_table.
			local label_part = sign_content.slice(this.prefix.len());
			local sign_tile = GSSign.GetLocation(sign);

			this.tile_table.rawset(label_part, sign_tile);

			// Remove sign
			GSSign.RemoveSign(sign);
		}
	}
}

function TileLabels::ExportToSave()
{
	return this.tile_table;
}

function TileLabels::ImportFromSave(save_data)
{
	// Ensure that we don't lose any labels read by ReadFromMap
	// by not doing a simple assignment of the save_data.
	//
	// If a label was read from the map, it takes precedence over
	// a label from the save game.
	foreach (key, val in save_data)
	{
		if (!this.tile_table.rawin(key))
			this.tile_table.rawset(key, val);
	}
}

